<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');

/**
* @package direct-as-a-service
* @subpackage models-folder
* @filesource
*/

/** */
require_once 'entity.php';
require_once 'mailbox.php';
require_once 'message.php';

/**
* @package direct-as-a-service
* @subpackage models
*/
class Folder extends Entity {
	static $database_group = 'mail_db';
	static $table = 'folders';	
	protected static $_relationships = array( 'mailbox' => array( 'type' => 'belongs_to'),
											  'message' => array( 'type' => 'has_many', 'condition' => 'sent=0 AND draft=0 AND archived=0'), //messages in folders must not be sent, drafts, or archives - shouldn't be able to create or update this, but enforce anyway
											 );
											 
	protected $_property_validation_rules = array( 'name' => 'nonempty_string',
												   'mailbox_id' => 'nonzero_unsigned_integer' );						 
	
	/**
	* Archive all the messages in this folder and then delete the folder.
	* @return boolean
	*/
	public function archive(){
		$mailbox = $this->mailbox;
		if(!$mailbox->is_active) return $this->error->warning("I can't archive ".$this->describe().' while '.$mailbox->describe().' is inactive');
		foreach($this->messages as $message){
			$message->archive();
		}
		return static::delete($this->id());	//foreign key constraint will cascade to null
	}	
	
	public function belongs_to_mailbox($mailbox){
		if(!Mailbox::is_an_entity($mailbox)) return $this->error->should_be_a_mailbox_entity($mailbox);
		$related_foreign_key = static::related_foreign_key('mailbox');
		return $this->$related_foreign_key == $mailbox->id();
	}		
	
	
///////////////////////
// GETTERS & SETTERS
///////////////////////	
	
	
//////////////////////////
// DATA MANAGEMENT
///////////////////////////	
	
	
	protected function _values_are_valid_for_create_and_update(){ 
		if(!parent::_values_are_valid_for_create_and_update()) return false;
		
		$mailbox_foreign_key = static::related_foreign_key('mailbox');
		
		//make sure that we're not trying to set name or mailbox to null
		foreach( array('name', $mailbox_foreign_key) as $required_field){
			if(!isset($this->$required_field)) return $this->error->warning('Unable to save '.$this->model_alias.' without a value for '.$this->model_alias.'::'.$required_field, 3);
			if($this->property_is_empty($required_field)) return $this->error->property_value_should_be_a_nonempty_string($required_field, get_class($this), $this->$required_field, 3);
		}
		
		//since we know we have a mailbox, make sure that it's an active mailbox
		$mailbox = $this->mailbox;
		
		if(!Mailbox::is_an_entity($mailbox)) 
			return $this->error->warning("I couldn't find a mailbox where ".$mailbox_foreign_key.'='.$this->$mailbox_foreign_key);
				
		if(!$mailbox->is_active)
			return $this->error->warning("I can't save changes to ".$this->describe().' for '.$mailbox->describe().' while '.$mailbox->describe().' is inactive');
		
		
#TODO - CAN WE LOWER OVERHEAD BY ONLY CHANGING THIS IF WE'VE CHANGED NAME/USERNAME?	
		//ensure that name is unique for the mailbox				
		$values = array('name' => $this->name, $mailbox_foreign_key => $this->$mailbox_foreign_key);
		if(!$this->property_is_empty('id'))
			$values['id !='] = $this->id();
		if(static::exists($values)) return $this->error->property_value_should_be_an_x('name', get_class($this), $this->name, 'unique name for mailbox#'.$this->error->describe($this->$mailbox_foreign_key), 2);		
		
		return true; 
	}
	
	function __get($property){
		
		if(string_begins_with('message_', $property)){
			//for message relationships, allow developers to append the name of a message part to the relationship name
			//e.g. $this->inbox_message_headers will return the headers for $this->inbox_messages
			$suffix = substr($property, strrpos($property, '_') + 1);
			if(Message::is_a_part($suffix)){
				$relationship = strip_from_end('_'.$suffix, $property);
				if(static::has_relationship($relationship) && static::relationships($relationship, 'model') == 'message'){
					$this->set_up_relationship_query($relationship);
					return Message::find_part($suffix);
				}
			}
		}
				
		return parent::__get($property);
	}
	
	function __call($name, $arguments){
		
		if(string_begins_with('message_', $name)){
			//for message relationships, allow developers to append the name of a message part to the relationship name
			//e.g. $this->inbox_message_headers() will return the headers for $this->inbox_messages()
			$suffix = substr($name, strrpos($name, '_') + 1);
			if(Message::is_a_part($suffix)){
				$relationship = strip_from_end('_'.$suffix, $name);
				if(static::has_relationship($relationship) && static::relationships($relationship, 'model') == 'message'){
					$this->set_up_relationship_query($relationship);
					return call_user_func_array('Message::find_part', array_merge( array($suffix), $arguments));
				}
			}
		}
				
		return parent::__call($name, $arguments);
	}		
	
}